"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.nonURIMatch = exports.default = void 0;

var _url = _interopRequireDefault(require("url"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Match and capture parts of a URI, like a specialized dialect of regular expression. This is used by PaneItem to
 * describe URIs that should launch specific panes.
 *
 * URI patterns used `{name}` placeholders to match any non-empty path segment or URI part (host, protocol) and capture
 * it as a parameter called "name". Any segment that is not recognized as a parameter will match exactly.
 *
 * Examples:
 *
 * `atom-github://hostname/exact/path?p0=value&p1=value` contains no parameters, so it will match _only_ that exact URL,
 * including query parameters and their values. Extra query parameters or a fragment (`#`) will cause the match to fail.
 *
 * `atom-github://hostname/path/{name}/fragment` will match and capture any second path segment.
 * * `atom-github://hostname/path/one/fragment` will match with `{name: 'one'}`
 * * `atom-github://hostname/path/two/fragment` will match with `{name: 'two'}`
 * * `atom-github://hostname/path/fragment` will not.
 *
 * `atom-github://hostname/root/{segments...}` will capture any number of path segments as an array. For example,
 * * `atom-github://hostname/root/foo/bar/baz/` will match with `{segments: ['foo', 'bar', 'baz']}`.
 * * `atom-github://hostname/root/single` will match with `{segments: ['single']}`; even a single segment will be
 *   matched as an array.
 * * `atom-github://hostname/root/` will match with `{segments: []}`.
 *
 * Query parameters and their values may be captured. Given: `atom-github://hostname?q={value}`
 * * `atom-github://hostname?q=foo` will match with `{value: 'foo'}`.
 * * `atom-github://hostname?q=one&q=two` will _not_ match.
 *
 * To match multiple query parameters, use a splat parameter. Given: `atom-github://hostname?q={value...}`
 * * `atom-github://hostname?q=one&q=two` will match with `{value: ['one', 'two']}`.
 * * `atom-github://hostname?q=single` will match with `{value: ['single']}`.
 * * `atom-github://hostname` will match with `{value: []}`.
 *
 * Protocol, username, password, or hostname may also contain capture expressions: `{p}://hostname`,
 * `foo://me:{password}@hostname`.
 */
class URIPattern {
  constructor(string) {
    this.original = string;

    const parsed = _url.default.parse(dashEscape(string), true);

    this.parts = {
      protocol: asPart(parsed.protocol, '', ':'),
      auth: splitAuth(parsed.auth, asPart),
      hostname: asPart(parsed.hostname),
      port: asPart(parsed.port),
      pathname: (parsed.pathname || '').split('/').slice(1).map(segment => asPart(segment)),
      query: Object.keys(parsed.query).reduce((acc, current) => {
        acc[current] = asPart(parsed.query[current]);
        return acc;
      }, {}),
      hash: asPart(parsed.hash, '#', '')
    };
  }

  matches(string) {
    if (string === undefined || string === null) {
      return nonURIMatch;
    }

    const other = _url.default.parse(string, true);

    const params = {}; // direct matches

    for (const attr of ['protocol', 'hostname', 'port', 'hash']) {
      if (!this.parts[attr].matchesIn(params, other[attr])) {
        return nonURIMatch;
      }
    } // auth


    const auth = splitAuth(other.auth);

    if (!this.parts.auth.username.matchesIn(params, auth.username)) {
      return nonURIMatch;
    }

    if (!this.parts.auth.password.matchesIn(params, auth.password)) {
      return nonURIMatch;
    } // pathname


    const pathParts = (other.pathname || '').split('/').filter(p => p.length > 0);
    let mineInd = 0;
    let yoursInd = 0;

    while (mineInd < this.parts.pathname.length && yoursInd < pathParts.length) {
      const mine = this.parts.pathname[mineInd];
      const yours = pathParts[yoursInd];

      if (!mine.matchesIn(params, yours)) {
        return nonURIMatch;
      } else {
        if (!mine.isSplat()) {
          mineInd++;
        }

        yoursInd++;
      }
    }

    while (mineInd < this.parts.pathname.length) {
      const part = this.parts.pathname[mineInd];

      if (!part.matchesEmptyIn(params)) {
        return nonURIMatch;
      }

      mineInd++;
    }

    if (yoursInd !== pathParts.length) {
      return nonURIMatch;
    } // query string


    const remaining = new Set(Object.keys(this.parts.query));

    for (const k in other.query) {
      const yours = other.query[k];
      remaining.delete(k);
      const mine = this.parts.query[k];

      if (mine === undefined) {
        return nonURIMatch;
      }

      const allYours = yours instanceof Array ? yours : [yours];

      for (const each of allYours) {
        if (!mine.matchesIn(params, each)) {
          return nonURIMatch;
        }
      }
    }

    for (const k of remaining) {
      const part = this.parts.query[k];

      if (!part.matchesEmptyIn(params)) {
        return nonURIMatch;
      }
    }

    return new URIMatch(string, params);
  } // Access the original string used to create this pattern.


  getOriginal() {
    return this.original;
  }

  toString() {
    return `<URIPattern ${this.original}>`;
  }

}
/**
 * Pattern component that matches its corresponding segment exactly.
 */


exports.default = URIPattern;

class ExactPart {
  constructor(string) {
    this.string = string;
  }

  matchesIn(params, other) {
    return other === this.string;
  }

  matchesEmptyIn(params) {
    return false;
  }

  isSplat() {
    return false;
  }

}
/**
 * Pattern component that matches and captures any non-empty corresponding segment within a URI.
 */


class CapturePart {
  constructor(name, splat, prefix, suffix) {
    this.name = name;
    this.splat = splat;
    this.prefix = prefix;
    this.suffix = suffix;
  }

  matchesIn(params, other) {
    if (this.prefix.length > 0 && other.startsWith(this.prefix)) {
      other = other.slice(this.prefix.length);
    }

    if (this.suffix.length > 0 && other.endsWith(this.suffix)) {
      other = other.slice(0, -this.suffix.length);
    }

    other = decodeURIComponent(other);

    if (this.name.length > 0) {
      if (this.splat) {
        if (params[this.name] === undefined) {
          params[this.name] = [other];
        } else {
          params[this.name].push(other);
        }
      } else {
        if (params[this.name] !== undefined) {
          return false;
        }

        params[this.name] = other;
      }
    }

    return true;
  }

  matchesEmptyIn(params) {
    if (this.splat) {
      if (params[this.name] === undefined) {
        params[this.name] = [];
      }

      return true;
    }

    return false;
  }

  isSplat() {
    return this.splat;
  }

}
/**
 * Including `{}` characters in certain URI components (hostname, protocol) cause `url.parse()` to lump everything into
 * the `pathname`. Escape brackets from a pattern with `-a` and `-z`, and literal dashes with `--`.
 */


function dashEscape(raw) {
  return raw.replace(/[{}-]/g, ch => {
    if (ch === '{') {
      return '-a';
    } else if (ch === '}') {
      return '-z';
    } else {
      return '--';
    }
  });
}
/**
 * Reverse the escaping performed by `dashEscape` by un-doubling `-` characters.
 */


function dashUnescape(escaped) {
  return escaped.replace(/--/g, '-');
}
/**
 * Parse a URI pattern component as either an `ExactPart` or a `CapturePart`. Recognize captures ending with `...` as
 * splat captures that can consume zero to many components.
 */


function asPart(patternSegment, prefix = '', suffix = '') {
  if (patternSegment === null) {
    return new ExactPart(null);
  }

  let subPattern = patternSegment;

  if (prefix.length > 0 && subPattern.startsWith(prefix)) {
    subPattern = subPattern.slice(prefix.length);
  }

  if (suffix.length > 0 && subPattern.endsWith(suffix)) {
    subPattern = subPattern.slice(0, -suffix.length);
  }

  if (subPattern.startsWith('-a') && subPattern.endsWith('-z')) {
    const splat = subPattern.endsWith('...-z');

    if (splat) {
      subPattern = subPattern.slice(2, -5);
    } else {
      subPattern = subPattern.slice(2, -2);
    }

    return new CapturePart(dashUnescape(subPattern), splat, prefix, suffix);
  } else {
    return new ExactPart(dashUnescape(patternSegment));
  }
}
/**
 * Split the `.auth` field into username and password subcomponent.
 */


function splitAuth(auth, fn = x => x) {
  if (auth === null) {
    return {
      username: fn(null),
      password: fn(null)
    };
  }

  const ind = auth.indexOf(':');
  return ind !== -1 ? {
    username: fn(auth.slice(0, ind)),
    password: fn(auth.slice(ind + 1))
  } : {
    username: fn(auth),
    password: fn(null)
  };
}
/**
 * Memorialize a successful match between a URI and a URIPattern, including any parameters that have been captured.
 */


class URIMatch {
  constructor(uri, params) {
    this.uri = uri;
    this.params = params;
  }

  ok() {
    return true;
  }

  getURI() {
    return this.uri;
  }

  getParams() {
    return this.params;
  }

  toString() {
    let s = '<URIMatch ok';

    for (const k in this.params) {
      s += ` ${k}="${this.params[k]}"`;
    }

    s += '>';
    return s;
  }

}
/**
 * Singleton object that memorializes an unsuccessful match between a URIPattern and an URI. Matches the API of a
 * URIMatch, but returns false for ok() and so on.
 */


const nonURIMatch = {
  ok() {
    return false;
  },

  getURI() {
    return undefined;
  },

  getParams() {
    return {};
  },

  toString() {
    return '<nonURIMatch>';
  }

};
exports.nonURIMatch = nonURIMatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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